package net.limosoft

import com.fasterxml.jackson.annotation.JsonIgnore
import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import com.fasterxml.jackson.annotation.JsonInclude
import grails.converters.JSON
import org.codehaus.groovy.grails.support.proxy.DefaultProxyHandler
import org.codehaus.groovy.grails.support.proxy.EntityProxyHandler
import org.codehaus.groovy.grails.support.proxy.ProxyHandler
import org.codehaus.groovy.grails.web.converters.ConverterUtil
import org.codehaus.groovy.grails.web.converters.exceptions.ConverterException
import org.codehaus.groovy.grails.web.converters.marshaller.ObjectMarshaller
import org.codehaus.groovy.grails.web.json.JSONWriter
import org.springframework.beans.BeanWrapper
import org.springframework.beans.BeanWrapperImpl

import java.lang.reflect.Field

import org.codehaus.groovy.grails.commons.*

/**
 * @author Dale "Ducky" Lotts
 * @author Siegfried Puchbauer
 * @since 2012/08/13
 */

public class DomainClassMarshaller implements ObjectMarshaller<JSON> {

    private GrailsApplication application;

    private boolean includeVersion = false;
    private ProxyHandler proxyHandler;

    public DomainClassMarshaller(boolean includeVersion, GrailsApplication application) {
        this(includeVersion, new DefaultProxyHandler(), application);
    }

    public DomainClassMarshaller(boolean includeVersion, ProxyHandler proxyHandler, GrailsApplication application) {
        this.includeVersion = includeVersion;
        this.proxyHandler = proxyHandler;
        this.application = application;
    }

    protected void asShortObject(Object refObj, JSON json, GrailsDomainClassProperty idProperty, GrailsDomainClass referencedDomainClass) throws ConverterException {
        Object idValue;

        if (proxyHandler instanceof EntityProxyHandler) {
            idValue = ((EntityProxyHandler) proxyHandler).getProxyIdentifier(refObj);
            if (idValue == null) {
                idValue = extractValue(refObj, idProperty);
            }
        }
        else {
            idValue = extractValue(refObj, idProperty);
        }
        JSONWriter writer = json.getWriter();
        writer.object();
        final clazz = referencedDomainClass.getClazz()

        if (!getIgnoredProperties(clazz).contains("class")) {
            writer.key("class").value(referencedDomainClass.getName());
        }

        writer.key("id").value(idValue);

        GrailsDomainClassProperty[] properties = referencedDomainClass.getPersistentProperties();

        Set<String> includeProperties = getIncludedProperties(clazz)

        BeanWrapper beanWrapper = new BeanWrapperImpl(refObj);

        for (GrailsDomainClassProperty property : properties) {
            if (includeProperties.contains(property.name)) {
                writeProperty(property, beanWrapper, json)
            }
        }

        writer.endObject();
    }

    protected Object extractValue(Object domainObject, GrailsDomainClassProperty property) {
        BeanWrapper beanWrapper = new BeanWrapperImpl(domainObject);
        return beanWrapper.getPropertyValue(property.getName());
    }

    private Set<String> getIgnoredProperties(Class<?> target) {
        Set<String> ignoredProperties = new HashSet<>();

        if (target.isAnnotationPresent(JsonIgnoreProperties)) {
            JsonIgnoreProperties annotation = target.getAnnotation(JsonIgnoreProperties)
            ignoredProperties.addAll(Arrays.asList(annotation.value()))
        }

        final ignored = target.declaredFields.findAll { field -> field.isAnnotationPresent(JsonIgnore)}
        for (Field field : ignored) {
            ignoredProperties.add(field.name)
        }
        ignoredProperties
    }

    private Set<String> getIncludedProperties(Class<?> target) {
        Set<String> includeProperties = new HashSet<>();
        final includes = target.declaredFields.findAll { field -> field.isAnnotationPresent(JsonInclude)}

        for (Field field : includes) {
            includeProperties.add(field.name)
        }
        includeProperties
    }

    public boolean isIncludeVersion() {
        return includeVersion;
    }

    protected boolean isRenderDomainClassRelations() {
        return false;
    }


    @SuppressWarnings([ "unchecked", "rawtypes" ])

    public void marshalObject(Object value, JSON json) throws ConverterException {
        JSONWriter writer = json.getWriter();
        value = proxyHandler.unwrapIfProxy(value);
        Class<?> clazz = value.getClass();

        GrailsDomainClass domainClass = (GrailsDomainClass)application.getArtefact(
              DomainClassArtefactHandler.TYPE, ConverterUtil.trimProxySuffix(clazz.getName()));
        BeanWrapper beanWrapper = new BeanWrapperImpl(value);

        writer.object();

        GrailsDomainClassProperty[] properties = domainClass.getPersistentProperties();

        Set<String> ignoredProperties = getIgnoredProperties(value.getClass())

        if (!ignoredProperties.contains("class")) {
            writer.key("class").value(domainClass.getClazz().getName())
        };

        GrailsDomainClassProperty id = domainClass.getIdentifier();
        Object idValue = extractValue(value, id);

        json.property("id", idValue);

        if (isIncludeVersion()) {
            GrailsDomainClassProperty versionProperty = domainClass.getVersion();
            Object version = extractValue(value, versionProperty);
            json.property("version", version);
        }

        for (GrailsDomainClassProperty property : properties) {
            if (!ignoredProperties.contains(property.name)) {
                writeProperty(property, beanWrapper, json)
            }
        }
        writer.endObject();
    }

    public boolean supports(Object object) {
        String name = ConverterUtil.trimProxySuffix(object.getClass().getName());
        return application.isArtefactOfType(DomainClassArtefactHandler.TYPE, name);
    }

    private void writeProperty(GrailsDomainClassProperty property, BeanWrapperImpl beanWrapper, JSON json) {
        json.getWriter().key(property.getName());
        if (!property.isAssociation()) {
            // Write non-relation property
            Object val = beanWrapper.getPropertyValue(property.getName());
            json.convertAnother(val);
        }
        else {
            Object referenceObject = beanWrapper.getPropertyValue(property.getName());
            if (isRenderDomainClassRelations()) {
                if (referenceObject == null) {
                    json.getWriter().value(null);
                }
                else {
                    referenceObject = proxyHandler.unwrapIfProxy(referenceObject);
                    if (referenceObject instanceof SortedMap) {
                        referenceObject = new TreeMap((SortedMap) referenceObject);
                    }
                    else if (referenceObject instanceof SortedSet) {
                        referenceObject = new TreeSet((SortedSet) referenceObject);
                    }
                    else if (referenceObject instanceof Set) {
                        referenceObject = new HashSet((Set) referenceObject);
                    }
                    else if (referenceObject instanceof Map) {
                        referenceObject = new HashMap((Map) referenceObject);
                    }
                    else if (referenceObject instanceof Collection) {
                        referenceObject = new ArrayList((Collection) referenceObject);
                    }
                    json.convertAnother(referenceObject);
                }
            }
            else {
                if (referenceObject == null) {
                    json.value(null);
                }
                else {
                    GrailsDomainClass referencedDomainClass = property.getReferencedDomainClass();

                    // Embedded are now always fully rendered
                    if (referencedDomainClass == null || property.isEmbedded() || GrailsClassUtils.isJdk5Enum(property.getType())) {
                        json.convertAnother(referenceObject);
                    }
                    else if (property.isOneToOne() || property.isManyToOne() || property.isEmbedded()) {
                        asShortObject(referenceObject, json, referencedDomainClass.getIdentifier(), referencedDomainClass);
                    }
                    else {
                        GrailsDomainClassProperty referencedIdProperty = referencedDomainClass.getIdentifier();
                        @SuppressWarnings("unused")
                        String refPropertyName = referencedDomainClass.getPropertyName();
                        if (referenceObject instanceof Collection) {
                            Collection o = (Collection) referenceObject;
                            json.getWriter().array();
                            for (Object el : o) {
                                asShortObject(el, json, referencedIdProperty, referencedDomainClass);
                            }
                            json.getWriter().endArray();
                        }
                        else if (referenceObject instanceof Map) {
                            Map<Object, Object> map = (Map<Object, Object>) referenceObject;
                            for (Map.Entry<Object, Object> entry : map.entrySet()) {
                                String key = String.valueOf(entry.getKey());
                                Object o = entry.getValue();
                                json.getWriter().object();
                                json.getWriter().key(key);
                                asShortObject(o, json, referencedIdProperty, referencedDomainClass);
                                json.getWriter().endObject();
                            }
                        }
                    }
                }
            }
        }
    }
}